<?php
define('PAGE_TITLE', 'Sales Invoices');
require_once '../config/config.php';
require_once '../libraries/Auth.php';

$auth = new Auth();
$auth->requireLogin();
$auth->requirePermission('sales');

include '../views/layouts/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <h1 class="h3 mb-0">Sales Invoices</h1>
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="<?php echo BASE_URL; ?>dashboard.php">Dashboard</a></li>
                <li class="breadcrumb-item"><a href="<?php echo BASE_URL; ?>sales/">Sales</a></li>
                <li class="breadcrumb-item active">Invoices</li>
            </ol>
        </nav>
    </div>
</div>

<!-- Filters -->
<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Filters</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-3">
                        <label class="form-label">Customer</label>
                        <select class="form-select" id="filterCustomer">
                            <option value="">All Customers</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Status</label>
                        <select class="form-select" id="filterStatus">
                            <option value="">All Status</option>
                            <option value="DRAFT">Draft</option>
                            <option value="SENT">Sent</option>
                            <option value="PAID">Paid</option>
                            <option value="OVERDUE">Overdue</option>
                            <option value="CANCELLED">Cancelled</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Date Range</label>
                        <input type="text" class="form-control" id="filterDateRange" placeholder="Select date range">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">&nbsp;</label>
                        <div>
                            <button class="btn btn-primary" onclick="applyFilters()">
                                <i class="fas fa-search"></i> Apply
                            </button>
                            <button class="btn btn-secondary" onclick="clearFilters()">
                                <i class="fas fa-undo"></i> Clear
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Invoices Table -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Invoices List</h5>
                <div>
                    <button class="btn btn-success" onclick="exportInvoices()">
                        <i class="fas fa-download me-2"></i>Export
                    </button>
                    <button class="btn btn-primary" onclick="openModal('addInvoice')">
                        <i class="fas fa-plus me-2"></i>New Invoice
                    </button>
                </div>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-hover" id="invoicesTable">
                        <thead>
                            <tr>
                                <th>Invoice #</th>
                                <th>Customer</th>
                                <th>Date</th>
                                <th>Due Date</th>
                                <th>Amount</th>
                                <th>Paid</th>
                                <th>Balance</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <!-- Data will be loaded via AJAX -->
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add/Edit Invoice Modal -->
<div class="modal fade" id="invoiceModal" tabindex="-1">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="invoiceModalTitle">New Sales Invoice</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="invoiceForm">
                    <input type="hidden" id="invoiceId" name="id">
                    
                    <div class="row">
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label class="form-label">Customer *</label>
                                <select class="form-select" id="invoiceCustomer" name="customer_id" required>
                                    <option value="">Select Customer</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label class="form-label">Invoice Date *</label>
                                <input type="date" class="form-control" id="invoiceDate" name="invoice_date" 
                                       value="<?php echo date('Y-m-d'); ?>" required>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label class="form-label">Due Date *</label>
                                <input type="date" class="form-control" id="invoiceDueDate" name="due_date" 
                                       value="<?php echo date('Y-m-d', strtotime('+30 days')); ?>" required>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Invoice #</label>
                                <input type="text" class="form-control" id="invoiceNumber" name="invoice_number" 
                                       placeholder="Auto-generated" readonly>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Reference</label>
                                <input type="text" class="form-control" id="invoiceReference" name="reference_no" 
                                       placeholder="Optional reference">
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Notes</label>
                        <textarea class="form-control" id="invoiceNotes" name="notes" rows="2"></textarea>
                    </div>
                    
                    <h6>Invoice Items</h6>
                    <div class="table-responsive">
                        <table class="table table-sm" id="invoiceItems">
                            <thead>
                                <tr>
                                    <th style="width: 35%">Item</th>
                                    <th style="width: 15%">Quantity</th>
                                    <th style="width: 15%">Price</th>
                                    <th style="width: 15%">Discount %</th>
                                    <th style="width: 15%">Total</th>
                                    <th style="width: 5%">Action</th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr>
                                    <td>
                                        <select class="form-select item-select" name="items[0][item_id]" required>
                                            <option value="">Select Item</option>
                                        </select>
                                    </td>
                                    <td>
                                        <input type="number" class="form-control item-quantity" 
                                               name="items[0][quantity]" min="0.01" value="1" step="0.01" required>
                                    </td>
                                    <td>
                                        <input type="number" class="form-control item-price" 
                                               name="items[0][unit_price]" min="0" step="0.01" required>
                                    </td>
                                    <td>
                                        <input type="number" class="form-control item-discount" 
                                               name="items[0][discount_percent]" min="0" max="100" value="0" step="0.01">
                                    </td>
                                    <td>
                                        <input type="number" class="form-control item-total" 
                                               name="items[0][total_price]" readonly>
                                    </td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-danger" onclick="removeItem(this)">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                        <button type="button" class="btn btn-sm btn-secondary" onclick="addItem()">
                            <i class="fas fa-plus"></i> Add Item
                        </button>
                    </div>
                    
                    <div class="row mt-3">
                        <div class="col-md-6 offset-md-6">
                            <table class="table table-sm">
                                <tr>
                                    <td class="text-end"><strong>Subtotal:</strong></td>
                                    <td class="text-end" style="width: 150px;">
                                        <span id="subtotalAmount">Rp 0</span>
                                    </td>
                                </tr>
                                <tr>
                                    <td class="text-end"><strong>Tax (10%):</strong></td>
                                    <td class="text-end">
                                        <span id="taxAmount">Rp 0</span>
                                    </td>
                                </tr>
                                <tr>
                                    <td class="text-end"><strong>Total:</strong></td>
                                    <td class="text-end">
                                        <h5 class="mb-0" id="totalAmount">Rp 0</h5>
                                    </td>
                                </tr>
                            </table>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="saveInvoice()">Save Invoice</button>
            </div>
        </div>
    </div>
</div>

<script>
// Load form data
function loadFormData() {
    // Load customers
    fetch('../api/sales/customers.php')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const customerSelect = document.getElementById('invoiceCustomer');
                const filterCustomer = document.getElementById('filterCustomer');
                
                customerSelect.innerHTML = '<option value="">Select Customer</option>';
                filterCustomer.innerHTML = '<option value="">All Customers</option>';
                
                data.customers.forEach(customer => {
                    const option = document.createElement('option');
                    option.value = customer.id;
                    option.textContent = `${customer.code} - ${customer.name}`;
                    
                    customerSelect.appendChild(option.cloneNode(true));
                    filterCustomer.appendChild(option.cloneNode(true));
                });
                
                initSelect2('#invoiceCustomer');
                initSelect2('#filterCustomer');
            }
        });
    
    // Load items for invoice items
    fetch('../api/inventory/active-items.php')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                window.items = data.items;
                updateItemSelects();
            }
        });
}

// Load invoices data
function loadInvoices() {
    const table = initDataTable('invoicesTable', {
        ajax: {
            url: '../api/sales/invoices-list.php',
            dataSrc: 'invoices'
        },
        columns: [
            { data: 'invoice_number' },
            { data: 'customer_name' },
            { 
                data: 'invoice_date',
                render: function(data) {
                    return formatDate(data);
                }
            },
            { 
                data: 'due_date',
                render: function(data) {
                    return formatDate(data);
                }
            },
            { 
                data: 'total_amount',
                render: function(data) {
                    return formatCurrency(data);
                }
            },
            { 
                data: 'paid_amount',
                render: function(data) {
                    return formatCurrency(data);
                }
            },
            { 
                data: null,
                render: function(data) {
                    const balance = data.total_amount - data.paid_amount;
                    return formatCurrency(balance);
                }
            },
            {
                data: 'status',
                render: function(data) {
                    const badges = {
                        'DRAFT': 'secondary',
                        'SENT': 'info',
                        'PAID': 'success',
                        'OVERDUE': 'warning',
                        'CANCELLED': 'danger'
                    };
                    return `<span class="badge bg-${badges[data]}">${data}</span>`;
                }
            },
            {
                data: null,
                render: function(data) {
                    return `
                        <button class="btn btn-sm btn-info" onclick="viewInvoice(${data.id})">
                            <i class="fas fa-eye"></i>
                        </button>
                        <button class="btn btn-sm btn-primary" onclick="editInvoice(${data.id})">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button class="btn btn-sm btn-success" onclick="recordPayment(${data.id})">
                            <i class="fas fa-money-bill"></i>
                        </button>
                        <button class="btn btn-sm btn-danger" onclick="deleteInvoice(${data.id})">
                            <i class="fas fa-trash"></i>
                        </button>
                    `;
                }
            }
        ]
    });
}

// Update item selects
function updateItemSelects() {
    const selects = document.querySelectorAll('.item-select');
    const options = window.items.map(item => 
        `<option value="${item.id}" data-price="${item.selling_price}">${item.name} (${item.code})</option>`
    ).join('');
    
    selects.forEach(select => {
        select.innerHTML = '<option value="">Select Item</option>' + options;
    });
}

// Apply filters
function applyFilters() {
    const table = $('#invoicesTable').DataTable();
    const customer = $('#filterCustomer').val();
    const status = $('#filterStatus').val();
    const dateRange = $('#filterDateRange').val();
    
    const params = new URLSearchParams();
    if (customer) params.append('customer', customer);
    if (status) params.append('status', status);
    if (dateRange) params.append('date_range', dateRange);
    
    table.ajax.url(`../api/sales/invoices-list.php?${params.toString()}`).load();
}

// Clear filters
function clearFilters() {
    $('#filterCustomer').val('');
    $('#filterStatus').val('');
    $('#filterDateRange').val('');
    
    const table = $('#invoicesTable').DataTable();
    table.ajax.url('../api/sales/invoices-list.php').load();
}

// Open modal for adding invoice
function openModal(type, invoiceId = null) {
    const modal = new bootstrap.Modal(document.getElementById('invoiceModal'));
    
    if (type === 'addInvoice') {
        document.getElementById('invoiceModalTitle').textContent = 'New Sales Invoice';
        document.getElementById('invoiceForm').reset();
        document.getElementById('invoiceId').value = '';
        
        // Generate invoice number
        fetch('../api/sales/next-invoice-number.php')
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    document.getElementById('invoiceNumber').value = data.invoice_number;
                }
            });
    } else if (type === 'editInvoice') {
        document.getElementById('invoiceModalTitle').textContent = 'Edit Invoice';
        loadInvoiceData(invoiceId);
    }
    
    modal.show();
}

// Calculate totals
function calculateTotals() {
    let subtotal = 0;
    
    document.querySelectorAll('#invoiceItems tbody tr').forEach(row => {
        const quantity = parseFloat(row.querySelector('.item-quantity').value) || 0;
        const price = parseFloat(row.querySelector('.item-price').value) || 0;
        const discount = parseFloat(row.querySelector('.item-discount').value) || 0;
        
        const lineTotal = quantity * price * (1 - discount / 100);
        row.querySelector('.item-total').value = lineTotal.toFixed(2);
        
        subtotal += lineTotal;
    });
    
    const tax = subtotal * 0.1; // 10% tax
    const total = subtotal + tax;
    
    document.getElementById('subtotalAmount').textContent = formatCurrency(subtotal);
    document.getElementById('taxAmount').textContent = formatCurrency(tax);
    document.getElementById('totalAmount').textContent = formatCurrency(total);
}

// Add item row
function addItem() {
    const tbody = document.querySelector('#invoiceItems tbody');
    const rowCount = tbody.rows.length;
    
    const newRow = tbody.insertRow();
    newRow.innerHTML = `
        <td>
            <select class="form-select item-select" name="items[${rowCount}][item_id]" required>
                <option value="">Select Item</option>
                ${window.items.map(item => 
                    `<option value="${item.id}" data-price="${item.selling_price}">${item.name} (${item.code})</option>`
                ).join('')}
            </select>
        </td>
        <td>
            <input type="number" class="form-control item-quantity" 
                   name="items[${rowCount}][quantity]" min="0.01" value="1" step="0.01" required>
        </td>
        <td>
            <input type="number" class="form-control item-price" 
                   name="items[${rowCount}][unit_price]" min="0" step="0.01" required>
        </td>
        <td>
            <input type="number" class="form-control item-discount" 
                   name="items[${rowCount}][discount_percent]" min="0" max="100" value="0" step="0.01">
        </td>
        <td>
            <input type="number" class="form-control item-total" 
                   name="items[${rowCount}][total_price]" readonly>
        </td>
        <td>
            <button type="button" class="btn btn-sm btn-danger" onclick="removeItem(this)">
                <i class="fas fa-trash"></i>
            </button>
        </td>
    `;
    
    // Add event listeners for new row
    const newInputs = newRow.querySelectorAll('input');
    newInputs.forEach(input => {
        input.addEventListener('input', calculateTotals);
    });
    
    const newSelect = newRow.querySelector('.item-select');
    newSelect.addEventListener('change', function() {
        const selectedOption = this.options[this.selectedIndex];
        const price = selectedOption.getAttribute('data-price');
        if (price) {
            newRow.querySelector('.item-price').value = price;
            calculateTotals();
        }
    });
}

// Remove item row
function removeItem(button) {
    const row = button.closest('tr');
    if (document.querySelectorAll('#invoiceItems tbody tr').length > 1) {
        row.remove();
        calculateTotals();
    }
}

// Save invoice
function saveInvoice() {
    const form = document.getElementById('invoiceForm');
    const formData = new FormData(form);
    
    fetch('../api/sales/save-invoice.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showSuccess('Invoice saved successfully!');
            $('#invoiceModal').modal('hide');
            $('#invoicesTable').DataTable().ajax.reload();
        } else {
            showError(data.message || 'Failed to save invoice');
        }
    });
}

// Initialize page
document.addEventListener('DOMContentLoaded', function() {
    loadFormData();
    loadInvoices();
    
    // Initialize date range picker
    initDateRangePicker('#filterDateRange', {
        autoUpdateInput: false,
        locale: {
            format: 'DD/MM/YYYY',
            cancelLabel: 'Clear'
        }
    });
    
    $('#filterDateRange').on('apply.daterangepicker', function(ev, picker) {
        $(this).val(picker.startDate.format('DD/MM/YYYY') + ' - ' + picker.endDate.format('DD/MM/YYYY'));
        applyFilters();
    });
    
    $('#filterDateRange').on('cancel.daterangepicker', function(ev, picker) {
        $(this).val('');
        applyFilters();
    });
    
    // Add event listeners for calculation
    document.addEventListener('input', function(e) {
        if (e.target.classList.contains('item-quantity') || 
            e.target.classList.contains('item-price') || 
            e.target.classList.contains('item-discount')) {
            calculateTotals();
        }
    });
});
</script>

<?php include '../views/layouts/footer.php'; ?>