<?php
define('PAGE_TITLE', 'Financial Reports');
require_once '../config/config.php';
require_once '../libraries/Auth.php';

$auth = new Auth();
$auth->requireLogin();
$auth->requirePermission('accounting');

include '../views/layouts/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <h1 class="h3 mb-0">Financial Reports</h1>
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="<?php echo BASE_URL; ?>dashboard.php">Dashboard</a></li>
                <li class="breadcrumb-item active">Financial Reports</li>
            </ol>
        </nav>
    </div>
</div>

<!-- Report Selection -->
<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Report Selection</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-3">
                        <div class="mb-3">
                            <label class="form-label">Report Type</label>
                            <select class="form-select" id="reportType" onchange="updateReportOptions()">
                                <option value="">Select Report</option>
                                <option value="profit-loss">Profit & Loss</option>
                                <option value="balance-sheet">Balance Sheet</option>
                                <option value="cash-flow">Cash Flow Statement</option>
                                <option value="trial-balance">Trial Balance</option>
                                <option value="general-ledger">General Ledger</option>
                                <option value="accounts-receivable">Accounts Receivable</option>
                                <option value="accounts-payable">Accounts Payable</option>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="mb-3">
                            <label class="form-label">Date Range</label>
                            <input type="text" class="form-control" id="reportDateRange" placeholder="Select date range">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="mb-3">
                            <label class="form-label">Format</label>
                            <select class="form-select" id="reportFormat">
                                <option value="pdf">PDF</option>
                                <option value="excel">Excel</option>
                                <option value="csv">CSV</option>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="mb-3">
                            <label class="form-label">&nbsp;</label>
                            <div>
                                <button class="btn btn-primary" onclick="generateReport()">
                                    <i class="fas fa-file-alt me-2"></i>Generate Report
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Report Preview -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">Report Preview</h5>
            </div>
            <div class="card-body">
                <div id="reportPreview">
                    <div class="text-center text-muted py-5">
                        <i class="fas fa-chart-line fa-3x mb-3"></i>
                        <h5>Select a report type and date range to preview</h5>
                        <p>Choose from various financial reports to analyze your business performance</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Profit & Loss Report Template -->
<div id="profitLossTemplate" style="display: none;">
    <div class="report-header">
        <h3 class="text-center">Profit & Loss Statement</h3>
        <p class="text-center text-muted">For the period: <span class="report-period"></span></p>
    </div>
    
    <div class="row">
        <div class="col-md-6">
            <h5>Revenue</h5>
            <table class="table table-sm">
                <thead>
                    <tr>
                        <th>Account</th>
                        <th class="text-end">Amount</th>
                    </tr>
                </thead>
                <tbody id="revenueAccounts">
                    <!-- Revenue accounts will be loaded here -->
                </tbody>
                <tfoot>
                    <tr class="table-primary">
                        <th>Total Revenue</th>
                        <th class="text-end" id="totalRevenue">Rp 0</th>
                    </tr>
                </tfoot>
            </table>
        </div>
        
        <div class="col-md-6">
            <h5>Expenses</h5>
            <table class="table table-sm">
                <thead>
                    <tr>
                        <th>Account</th>
                        <th class="text-end">Amount</th>
                    </tr>
                </thead>
                <tbody id="expenseAccounts">
                    <!-- Expense accounts will be loaded here -->
                </tbody>
                <tfoot>
                    <tr class="table-danger">
                        <th>Total Expenses</th>
                        <th class="text-end" id="totalExpenses">Rp 0</th>
                    </tr>
                </tfoot>
            </table>
        </div>
    </div>
    
    <div class="row">
        <div class="col-12">
            <table class="table table-sm">
                <tfoot>
                    <tr class="table-success">
                        <th class="text-end">Net Profit</th>
                        <th class="text-end" style="width: 200px;" id="netProfit">Rp 0</th>
                    </tr>
                </tfoot>
            </table>
        </div>
    </div>
</div>

<script>
// Initialize date range picker
document.addEventListener('DOMContentLoaded', function() {
    initDateRangePicker('#reportDateRange', {
        ranges: {
            'This Month': [moment().startOf('month'), moment().endOf('month')],
            'Last Month': [moment().subtract(1, 'month').startOf('month'), moment().subtract(1, 'month').endOf('month')],
            'This Quarter': [moment().startOf('quarter'), moment().endOf('quarter')],
            'This Year': [moment().startOf('year'), moment().endOf('year')],
            'Last Year': [moment().subtract(1, 'year').startOf('year'), moment().subtract(1, 'year').endOf('year')]
        },
        locale: {
            format: 'DD/MM/YYYY'
        }
    });
});

// Update report options based on type
function updateReportOptions() {
    const reportType = document.getElementById('reportType').value;
    const reportPreview = document.getElementById('reportPreview');
    
    switch(reportType) {
        case 'profit-loss':
            loadProfitLossReport();
            break;
        case 'balance-sheet':
            loadBalanceSheetReport();
            break;
        case 'cash-flow':
            loadCashFlowReport();
            break;
        case 'trial-balance':
            loadTrialBalanceReport();
            break;
        case 'general-ledger':
            loadGeneralLedgerReport();
            break;
        case 'accounts-receivable':
            loadAccountsReceivableReport();
            break;
        case 'accounts-payable':
            loadAccountsPayableReport();
            break;
        default:
            reportPreview.innerHTML = `
                <div class="text-center text-muted py-5">
                    <i class="fas fa-chart-line fa-3x mb-3"></i>
                    <h5>Select a report type and date range to preview</h5>
                    <p>Choose from various financial reports to analyze your business performance</p>
                </div>
            `;
    }
}

// Load Profit & Loss Report
function loadProfitLossReport() {
    const dateRange = document.getElementById('reportDateRange').val();
    if (!dateRange) {
        showWarning('Please select a date range');
        return;
    }
    
    showLoading();
    
    fetch(`../api/reports/profit-loss.php?date_range=${dateRange}`)
        .then(response => response.json())
        .then(data => {
            hideLoading();
            if (data.success) {
                displayProfitLossReport(data.report);
            } else {
                showError(data.message || 'Failed to load report');
            }
        });
}

// Display Profit & Loss Report
function displayProfitLossReport(report) {
    const template = document.getElementById('profitLossTemplate').cloneNode(true);
    template.style.display = 'block';
    
    template.querySelector('.report-period').textContent = report.period;
    
    // Revenue accounts
    const revenueTbody = template.querySelector('#revenueAccounts');
    revenueTbody.innerHTML = '';
    report.revenue.forEach(account => {
        const row = document.createElement('tr');
        row.innerHTML = `
            <td>${account.account_name}</td>
            <td class="text-end">${formatCurrency(account.amount)}</td>
        `;
        revenueTbody.appendChild(row);
    });
    
    // Expense accounts
    const expenseTbody = template.querySelector('#expenseAccounts');
    expenseTbody.innerHTML = '';
    report.expenses.forEach(account => {
        const row = document.createElement('tr');
        row.innerHTML = `
            <td>${account.account_name}</td>
            <td class="text-end">${formatCurrency(account.amount)}</td>
        `;
        expenseTbody.appendChild(row);
    });
    
    // Totals
    template.querySelector('#totalRevenue').textContent = formatCurrency(report.total_revenue);
    template.querySelector('#totalExpenses').textContent = formatCurrency(report.total_expenses);
    template.querySelector('#netProfit').textContent = formatCurrency(report.net_profit);
    
    document.getElementById('reportPreview').innerHTML = '';
    document.getElementById('reportPreview').appendChild(template);
}

// Generate report file
function generateReport() {
    const reportType = document.getElementById('reportType').value;
    const dateRange = document.getElementById('reportDateRange').value;
    const format = document.getElementById('reportFormat').value;
    
    if (!reportType) {
        showWarning('Please select a report type');
        return;
    }
    
    if (!dateRange) {
        showWarning('Please select a date range');
        return;
    }
    
    const params = new URLSearchParams({
        type: reportType,
        date_range: dateRange,
        format: format
    });
    
    window.open(`../api/reports/generate-report.php?${params.toString()}`, '_blank');
}

// Load other report types (implement as needed)
function loadBalanceSheetReport() {
    // Implementation for balance sheet
    document.getElementById('reportPreview').innerHTML = `
        <div class="text-center py-5">
            <h5>Balance Sheet Report</h5>
            <p>Coming soon...</p>
        </div>
    `;
}

function loadCashFlowReport() {
    // Implementation for cash flow
    document.getElementById('reportPreview').innerHTML = `
        <div class="text-center py-5">
            <h5>Cash Flow Statement</h5>
            <p>Coming soon...</p>
        </div>
    `;
}

function loadTrialBalanceReport() {
    // Implementation for trial balance
    document.getElementById('reportPreview').innerHTML = `
        <div class="text-center py-5">
            <h5>Trial Balance</h5>
            <p>Coming soon...</p>
        </div>
    `;
}

function loadGeneralLedgerReport() {
    // Implementation for general ledger
    document.getElementById('reportPreview').innerHTML = `
        <div class="text-center py-5">
            <h5>General Ledger</h5>
            <p>Coming soon...</p>
        </div>
    `;
}

function loadAccountsReceivableReport() {
    // Implementation for accounts receivable
    document.getElementById('reportPreview').innerHTML = `
        <div class="text-center py-5">
            <h5>Accounts Receivable Report</h5>
            <p>Coming soon...</p>
        </div>
    `;
}

function loadAccountsPayableReport() {
    // Implementation for accounts payable
    document.getElementById('reportPreview').innerHTML = `
        <div class="text-center py-5">
            <h5>Accounts Payable Report</h5>
            <p>Coming soon...</p>
        </div>
    `;
}
</script>

<?php include '../views/layouts/footer.php'; ?>