<?php
define('PAGE_TITLE', 'Employee Management');
require_once '../config/config.php';
require_once '../libraries/Auth.php';

$auth = new Auth();
$auth->requireLogin();
$auth->requirePermission('hr');

include '../views/layouts/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <h1 class="h3 mb-0">Employee Management</h1>
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="<?php echo BASE_URL; ?>dashboard.php">Dashboard</a></li>
                <li class="breadcrumb-item"><a href="<?php echo BASE_URL; ?>payroll/">Payroll</a></li>
                <li class="breadcrumb-item active">Employees</li>
            </ol>
        </nav>
    </div>
</div>

<!-- Employee List -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Employee List</h5>
                <div>
                    <button class="btn btn-success" onclick="exportEmployees()">
                        <i class="fas fa-download me-2"></i>Export
                    </button>
                    <button class="btn btn-primary" onclick="openModal('addEmployee')">
                        <i class="fas fa-plus me-2"></i>Add Employee
                    </button>
                </div>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-hover" id="employeesTable">
                        <thead>
                            <tr>
                                <th>Code</th>
                                <th>Name</th>
                                <th>Position</th>
                                <th>Department</th>
                                <th>Email</th>
                                <th>Phone</th>
                                <th>Basic Salary</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <!-- Data will be loaded via AJAX -->
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add/Edit Employee Modal -->
<div class="modal fade" id="employeeModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="employeeModalTitle">Add Employee</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="employeeForm">
                    <input type="hidden" id="employeeId" name="id">
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Employee Code *</label>
                                <input type="text" class="form-control" id="employeeCode" name="employee_code" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Full Name *</label>
                                <input type="text" class="form-control" id="employeeName" name="full_name" required>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Email</label>
                                <input type="email" class="form-control" id="employeeEmail" name="email">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Phone</label>
                                <input type="tel" class="form-control" id="employeePhone" name="phone">
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Position</label>
                                <input type="text" class="form-control" id="employeePosition" name="position">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Department</label>
                                <input type="text" class="form-control" id="employeeDepartment" name="department">
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Date of Birth</label>
                                <input type="date" class="form-control" id="employeeDOB" name="date_of_birth">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Date Joined *</label>
                                <input type="date" class="form-control" id="employeeJoined" name="date_joined" required>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Basic Salary *</label>
                                <input type="number" class="form-control" id="employeeSalary" name="basic_salary" 
                                       min="0" step="1000" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Allowance</label>
                                <input type="number" class="form-control" id="employeeAllowance" name="allowance" 
                                       min="0" step="1000" value="0">
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Tax Status</label>
                                <select class="form-select" id="employeeTaxStatus" name="tax_status">
                                    <option value="TK/0">TK/0</option>
                                    <option value="TK/1">TK/1</option>
                                    <option value="TK/2">TK/2</option>
                                    <option value="TK/3">TK/3</option>
                                    <option value="K/0">K/0</option>
                                    <option value="K/1">K/1</option>
                                    <option value="K/2">K/2</option>
                                    <option value="K/3">K/3</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Bank Account</label>
                                <input type="text" class="form-control" id="employeeBankAccount" name="bank_account" 
                                       placeholder="Bank account number">
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Address</label>
                        <textarea class="form-control" id="employeeAddress" name="address" rows="3"></textarea>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Status</label>
                        <select class="form-select" id="employeeStatus" name="is_active">
                            <option value="1">Active</option>
                            <option value="0">Inactive</option>
                        </select>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="saveEmployee()">Save Employee</button>
            </div>
        </div>
    </div>
</div>

<script>
// Load employees data
function loadEmployees() {
    const table = initDataTable('employeesTable', {
        ajax: {
            url: '../api/payroll/employees-list.php',
            dataSrc: 'employees'
        },
        columns: [
            { data: 'employee_code' },
            { data: 'full_name' },
            { data: 'position' },
            { data: 'department' },
            { data: 'email' },
            { data: 'phone' },
            { 
                data: 'basic_salary',
                render: function(data) {
                    return formatCurrency(data);
                }
            },
            {
                data: 'is_active',
                render: function(data) {
                    return data == 1 ? 
                        '<span class="badge bg-success">Active</span>' : 
                        '<span class="badge bg-secondary">Inactive</span>';
                }
            },
            {
                data: null,
                render: function(data) {
                    return `
                        <button class="btn btn-sm btn-info" onclick="viewEmployee(${data.id})">
                            <i class="fas fa-eye"></i>
                        </button>
                        <button class="btn btn-sm btn-primary" onclick="editEmployee(${data.id})">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button class="btn btn-sm btn-danger" onclick="deleteEmployee(${data.id})">
                            <i class="fas fa-trash"></i>
                        </button>
                    `;
                }
            }
        ]
    });
}

// Open modal for adding/editing employee
function openModal(type, employeeId = null) {
    const modal = new bootstrap.Modal(document.getElementById('employeeModal'));
    
    if (type === 'addEmployee') {
        document.getElementById('employeeModalTitle').textContent = 'Add Employee';
        document.getElementById('employeeForm').reset();
        document.getElementById('employeeId').value = '';
        document.getElementById('employeeJoined').value = new Date().toISOString().split('T')[0];
    } else if (type === 'editEmployee') {
        document.getElementById('employeeModalTitle').textContent = 'Edit Employee';
        loadEmployeeData(employeeId);
    }
    
    modal.show();
}

// Load employee data for editing
function loadEmployeeData(employeeId) {
    fetch(`../api/payroll/get-employee.php?id=${employeeId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const employee = data.employee;
                document.getElementById('employeeId').value = employee.id;
                document.getElementById('employeeCode').value = employee.employee_code;
                document.getElementById('employeeName').value = employee.full_name;
                document.getElementById('employeeEmail').value = employee.email || '';
                document.getElementById('employeePhone').value = employee.phone || '';
                document.getElementById('employeePosition').value = employee.position || '';
                document.getElementById('employeeDepartment').value = employee.department || '';
                document.getElementById('employeeDOB').value = employee.date_of_birth || '';
                document.getElementById('employeeJoined').value = employee.date_joined;
                document.getElementById('employeeSalary').value = employee.basic_salary;
                document.getElementById('employeeAllowance').value = employee.allowance || 0;
                document.getElementById('employeeTaxStatus').value = employee.tax_status || 'TK/0';
                document.getElementById('employeeBankAccount').value = employee.bank_account || '';
                document.getElementById('employeeAddress').value = employee.address || '';
                document.getElementById('employeeStatus').value = employee.is_active;
            }
        });
}

// Save employee
function saveEmployee() {
    const form = document.getElementById('employeeForm');
    const formData = new FormData(form);
    
    fetch('../api/payroll/save-employee.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showSuccess('Employee saved successfully!');
            $('#employeeModal').modal('hide');
            $('#employeesTable').DataTable().ajax.reload();
        } else {
            showError(data.message || 'Failed to save employee');
        }
    });
}

// Delete employee
function deleteEmployee(employeeId) {
    confirmDialog('Delete Employee', 'Are you sure you want to delete this employee?', function() {
        fetch('../api/payroll/delete-employee.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ id: employeeId })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showSuccess('Employee deleted successfully!');
                $('#employeesTable').DataTable().ajax.reload();
            } else {
                showError(data.message || 'Failed to delete employee');
            }
        });
    });
}

// View employee details
function viewEmployee(employeeId) {
    window.location.href = `employee-detail.php?id=${employeeId}`;
}

// Export employees
function exportEmployees() {
    window.location.href = '../api/payroll/export-employees.php';
}

// Initialize page
document.addEventListener('DOMContentLoaded', function() {
    loadEmployees();
});
</script>

<?php include '../views/layouts/footer.php'; ?>