<?php
class Auth {
    private $db;
    
    public function __construct() {
        $this->db = new Database();
    }
    
    public function login($username, $password) {
        $this->db->query('SELECT u.*, r.name as role_name, r.permissions, c.name as company_name 
                          FROM users u 
                          LEFT JOIN roles r ON u.role_id = r.id 
                          LEFT JOIN companies c ON u.company_id = c.id 
                          WHERE (u.username = :username OR u.email = :username) 
                          AND u.is_active = 1');
        
        $this->db->bind(':username', $username);
        $user = $this->db->single();
        
        if ($user && password_verify($password, $user['password_hash'])) {
            // Generate session token
            $sessionToken = bin2hex(random_bytes(32));
            
            // Store session
            $this->db->query('INSERT INTO user_sessions (user_id, session_token, ip_address, user_agent, expires_at) 
                              VALUES (:user_id, :token, :ip, :agent, DATE_ADD(NOW(), INTERVAL :timeout SECOND))');
            
            $this->db->bind(':user_id', $user['id']);
            $this->db->bind(':token', $sessionToken);
            $this->db->bind(':ip', $_SERVER['REMOTE_ADDR']);
            $this->db->bind(':agent', $_SERVER['HTTP_USER_AGENT'] ?? '');
            $this->db->bind(':timeout', SESSION_TIMEOUT);
            $this->db->execute();
            
            // Update last login
            $this->db->query('UPDATE users SET last_login = NOW() WHERE id = :id');
            $this->db->bind(':id', $user['id']);
            $this->db->execute();
            
            // Set session variables
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['full_name'] = $user['full_name'];
            $_SESSION['role'] = $user['role_name'];
            $_SESSION['permissions'] = json_decode($user['permissions'], true);
            $_SESSION['company_id'] = $user['company_id'];
            $_SESSION['company_name'] = $user['company_name'];
            $_SESSION['session_token'] = $sessionToken;
            
            return true;
        }
        
        return false;
    }
    
    public function logout() {
        if (isset($_SESSION['session_token'])) {
            $this->db->query('DELETE FROM user_sessions WHERE session_token = :token');
            $this->db->bind(':token', $_SESSION['session_token']);
            $this->db->execute();
        }
        
        session_destroy();
    }
    
    public function isLoggedIn() {
        if (!isset($_SESSION['user_id']) || !isset($_SESSION['session_token'])) {
            return false;
        }
        
        // Validate session token
        $this->db->query('SELECT * FROM user_sessions 
                          WHERE session_token = :token 
                          AND expires_at > NOW()');
        
        $this->db->bind(':token', $_SESSION['session_token']);
        $session = $this->db->single();
        
        if (!$session) {
            $this->logout();
            return false;
        }
        
        // Extend session
        $this->db->query('UPDATE user_sessions 
                          SET expires_at = DATE_ADD(NOW(), INTERVAL :timeout SECOND) 
                          WHERE session_token = :token');
        
        $this->db->bind(':timeout', SESSION_TIMEOUT);
        $this->db->bind(':token', $_SESSION['session_token']);
        $this->db->execute();
        
        return true;
    }
    
    public function hasPermission($permission) {
        if (!isset($_SESSION['permissions'])) {
            return false;
        }
        
        $permissions = $_SESSION['permissions'];
        
        // Super admin has all permissions
        if (isset($permissions['all']) && $permissions['all'] === true) {
            return true;
        }
        
        return isset($permissions[$permission]) && $permissions[$permission] === true;
    }
    
    public function requireLogin() {
        if (!$this->isLoggedIn()) {
            header('Location: ' . BASE_URL . 'login.php');
            exit();
        }
    }
    
    public function requirePermission($permission) {
        $this->requireLogin();
        
        if (!$this->hasPermission($permission)) {
            header('HTTP/1.1 403 Forbidden');
            die('Access Denied: Insufficient permissions');
        }
    }
    
    public function getCurrentUser() {
        if (!$this->isLoggedIn()) {
            return null;
        }
        
        $this->db->query('SELECT * FROM users WHERE id = :id');
        $this->db->bind(':id', $_SESSION['user_id']);
        return $this->db->single();
    }
}
?>