<?php
require_once '../../config/config.php';
require_once '../../config/database.php';
require_once '../../libraries/Auth.php';

header('Content-Type: application/json');

$auth = new Auth();
if (!$auth->isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized']);
    exit();
}

if (!$auth->hasPermission('accounting')) {
    http_response_code(403);
    echo json_encode(['error' => 'Forbidden']);
    exit();
}

$db = new Database();
$companyId = $_SESSION['company_id'] ?? 1;

try {
    $id = $_POST['id'] ?? null;
    $accountCode = $_POST['account_code'] ?? '';
    $accountName = $_POST['account_name'] ?? '';
    $accountTypeId = $_POST['account_type_id'] ?? '';
    $parentAccountId = $_POST['parent_account_id'] ?? null;
    $openingBalance = $_POST['opening_balance'] ?? 0;
    $isActive = $_POST['is_active'] ?? 1;
    $description = $_POST['description'] ?? '';
    
    // Validation
    if (empty($accountCode) || empty($accountName) || empty($accountTypeId)) {
        throw new Exception('Required fields are missing');
    }
    
    if ($parentAccountId === '') {
        $parentAccountId = null;
    }
    
    $db->beginTransaction();
    
    if ($id) {
        // Update existing account
        $db->query("
            UPDATE chart_of_accounts 
            SET account_code = :code,
                account_name = :name,
                account_type_id = :type_id,
                parent_account_id = :parent_id,
                opening_balance = :opening_balance,
                current_balance = current_balance + (:opening_balance - opening_balance),
                is_active = :is_active,
                description = :description
            WHERE id = :id AND company_id = :company_id
        ");
        
        $db->bind(':id', $id);
        $db->bind(':code', $accountCode);
        $db->bind(':name', $accountName);
        $db->bind(':type_id', $accountTypeId);
        $db->bind(':parent_id', $parentAccountId);
        $db->bind(':opening_balance', $openingBalance);
        $db->bind(':is_active', $isActive);
        $db->bind(':description', $description);
        $db->bind(':company_id', $companyId);
        
    } else {
        // Insert new account
        $db->query("
            INSERT INTO chart_of_accounts 
            (company_id, account_code, account_name, account_type_id, parent_account_id, 
             opening_balance, current_balance, is_active, description)
            VALUES 
            (:company_id, :code, :name, :type_id, :parent_id, :opening_balance, 
             :opening_balance, :is_active, :description)
        ");
        
        $db->bind(':company_id', $companyId);
        $db->bind(':code', $accountCode);
        $db->bind(':name', $accountName);
        $db->bind(':type_id', $accountTypeId);
        $db->bind(':parent_id', $parentAccountId);
        $db->bind(':opening_balance', $openingBalance);
        $db->bind(':is_active', $isActive);
        $db->bind(':description', $description);
    }
    
    $db->execute();
    
    // Create journal entry for opening balance if new account
    if (!$id && $openingBalance != 0) {
        $accountId = $id ? $id : $db->lastInsertId();
        
        // Determine if debit or credit based on account type
        $db->query("SELECT normal_balance FROM account_types WHERE id = :type_id");
        $db->bind(':type_id', $accountTypeId);
        $accountType = $db->single();
        
        if ($accountType) {
            $normalBalance = $accountType['normal_balance'];
            
            $db->query("
                INSERT INTO journal_entries 
                (company_id, transaction_no, transaction_date, description, total_amount, created_by)
                VALUES 
                (:company_id, :transaction_no, CURDATE(), :description, :amount, :created_by)
            ");
            
            $transactionNo = 'OPN-' . date('Ymd') . '-' . $accountId;
            $description = "Opening balance for account $accountName";
            
            $db->bind(':company_id', $companyId);
            $db->bind(':transaction_no', $transactionNo);
            $db->bind(':description', $description);
            $db->bind(':amount', abs($openingBalance));
            $db->bind(':created_by', $_SESSION['user_id']);
            
            $db->execute();
            
            $journalEntryId = $db->lastInsertId();
            
            // Create journal entry details
            $db->query("
                INSERT INTO journal_entry_details 
                (journal_entry_id, account_id, description, debit_amount, credit_amount)
                VALUES 
                (:journal_id, :account_id, :description, :debit, :credit)
            ");
            
            if ($normalBalance === 'debit') {
                $db->bind(':debit', $openingBalance);
                $db->bind(':credit', 0);
            } else {
                $db->bind(':debit', 0);
                $db->bind(':credit', $openingBalance);
            }
            
            $db->bind(':journal_id', $journalEntryId);
            $db->bind(':account_id', $accountId);
            $db->bind(':description', $description);
            $db->execute();
            
            // Balance with opening balance equity account
            $db->query("SELECT id FROM chart_of_accounts 
                        WHERE account_code = '3-1100' AND company_id = :company_id");
            $db->bind(':company_id', $companyId);
            $equityAccount = $db->single();
            
            if ($equityAccount) {
                $db->query("
                    INSERT INTO journal_entry_details 
                    (journal_entry_id, account_id, description, debit_amount, credit_amount)
                    VALUES 
                    (:journal_id, :account_id, :description, :debit, :credit)
                ");
                
                if ($normalBalance === 'debit') {
                    $db->bind(':debit', 0);
                    $db->bind(':credit', $openingBalance);
                } else {
                    $db->bind(':debit', $openingBalance);
                    $db->bind(':credit', 0);
                }
                
                $db->bind(':journal_id', $journalEntryId);
                $db->bind(':account_id', $equityAccount['id']);
                $db->bind(':description', $description);
                $db->execute();
            }
        }
    }
    
    $db->endTransaction();
    
    echo json_encode([
        'success' => true,
        'message' => 'Account saved successfully'
    ]);
    
} catch (Exception $e) {
    if (isset($db)) {
        $db->cancelTransaction();
    }
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>