<?php
define('PAGE_TITLE', 'Chart of Accounts');
require_once '../config/config.php';
require_once '../libraries/Auth.php';

$auth = new Auth();
$auth->requireLogin();
$auth->requirePermission('accounting');

include '../views/layouts/header.php';
?>

<div class="row mb-4">
    <div class="col-12">
        <h1 class="h3 mb-0">Chart of Accounts</h1>
        <nav aria-label="breadcrumb">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="<?php echo BASE_URL; ?>dashboard.php">Dashboard</a></li>
                <li class="breadcrumb-item active">Chart of Accounts</li>
            </ol>
        </nav>
    </div>
</div>

<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0">Account List</h5>
                <button class="btn btn-primary" onclick="openModal('addAccount')">
                    <i class="fas fa-plus me-2"></i>Add Account
                </button>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-hover" id="coaTable">
                        <thead>
                            <tr>
                                <th>Account Code</th>
                                <th>Account Name</th>
                                <th>Type</th>
                                <th>Category</th>
                                <th>Balance</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <!-- Data will be loaded via AJAX -->
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add/Edit Account Modal -->
<div class="modal fade" id="accountModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="accountModalTitle">Add Account</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="accountForm">
                    <input type="hidden" id="accountId" name="id">
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Account Code *</label>
                                <input type="text" class="form-control" id="accountCode" name="account_code" required>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Account Name *</label>
                                <input type="text" class="form-control" id="accountName" name="account_name" required>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Account Type *</label>
                                <select class="form-select" id="accountType" name="account_type_id" required>
                                    <option value="">Select Type</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Parent Account</label>
                                <select class="form-select" id="parentAccount" name="parent_account_id">
                                    <option value="">No Parent</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Opening Balance</label>
                                <input type="number" class="form-control" id="openingBalance" name="opening_balance" step="0.01" value="0">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Status</label>
                                <select class="form-select" id="accountStatus" name="is_active">
                                    <option value="1">Active</option>
                                    <option value="0">Inactive</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Description</label>
                        <textarea class="form-control" id="accountDescription" name="description" rows="3"></textarea>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="saveAccount()">Save Account</button>
            </div>
        </div>
    </div>
</div>

<script>
// Load account types and parent accounts
function loadFormData() {
    // Load account types
    fetch('../api/accounting/account-types.php')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const accountTypeSelect = document.getElementById('accountType');
                accountTypeSelect.innerHTML = '<option value="">Select Type</option>';
                
                data.account_types.forEach(type => {
                    const option = document.createElement('option');
                    option.value = type.id;
                    option.textContent = `${type.code} - ${type.name}`;
                    accountTypeSelect.appendChild(option);
                });
            }
        });
    
    // Load parent accounts
    fetch('../api/accounting/parent-accounts.php')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const parentSelect = document.getElementById('parentAccount');
                parentSelect.innerHTML = '<option value="">No Parent</option>';
                
                data.parent_accounts.forEach(account => {
                    const option = document.createElement('option');
                    option.value = account.id;
                    option.textContent = `${account.account_code} - ${account.account_name}`;
                    parentSelect.appendChild(option);
                });
            }
        });
}

// Load COA data
function loadCOA() {
    const table = initDataTable('coaTable', {
        ajax: {
            url: '../api/accounting/coa-list.php',
            dataSrc: 'accounts'
        },
        columns: [
            { data: 'account_code' },
            { data: 'account_name' },
            { data: 'type_name' },
            { data: 'category' },
            { 
                data: 'current_balance',
                render: function(data) {
                    return formatCurrency(data);
                }
            },
            {
                data: 'is_active',
                render: function(data) {
                    return data == 1 ? 
                        '<span class="badge bg-success">Active</span>' : 
                        '<span class="badge bg-secondary">Inactive</span>';
                }
            },
            {
                data: null,
                render: function(data) {
                    return `
                        <button class="btn btn-sm btn-primary" onclick="editAccount(${data.id})">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button class="btn btn-sm btn-danger" onclick="deleteAccount(${data.id})">
                            <i class="fas fa-trash"></i>
                        </button>
                    `;
                }
            }
        ]
    });
}

// Open modal for adding/editing account
function openModal(type, accountId = null) {
    const modal = new bootstrap.Modal(document.getElementById('accountModal'));
    
    if (type === 'addAccount') {
        document.getElementById('accountModalTitle').textContent = 'Add Account';
        document.getElementById('accountForm').reset();
        document.getElementById('accountId').value = '';
    } else if (type === 'editAccount') {
        document.getElementById('accountModalTitle').textContent = 'Edit Account';
        loadAccountData(accountId);
    }
    
    modal.show();
}

// Load account data for editing
function loadAccountData(accountId) {
    fetch(`../api/accounting/get-account.php?id=${accountId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const account = data.account;
                document.getElementById('accountId').value = account.id;
                document.getElementById('accountCode').value = account.account_code;
                document.getElementById('accountName').value = account.account_name;
                document.getElementById('accountType').value = account.account_type_id;
                document.getElementById('parentAccount').value = account.parent_account_id || '';
                document.getElementById('openingBalance').value = account.opening_balance;
                document.getElementById('accountStatus').value = account.is_active;
                document.getElementById('accountDescription').value = account.description || '';
            }
        });
}

// Save account
function saveAccount() {
    const form = document.getElementById('accountForm');
    const formData = new FormData(form);
    
    fetch('../api/accounting/save-account.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showSuccess('Account saved successfully!');
            $('#accountModal').modal('hide');
            $('#coaTable').DataTable().ajax.reload();
        } else {
            showError(data.message || 'Failed to save account');
        }
    });
}

// Delete account
function deleteAccount(accountId) {
    confirmDialog('Delete Account', 'Are you sure you want to delete this account?', function() {
        fetch('../api/accounting/delete-account.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({ id: accountId })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showSuccess('Account deleted successfully!');
                $('#coaTable').DataTable().ajax.reload();
            } else {
                showError(data.message || 'Failed to delete account');
            }
        });
    });
}

// Initialize page
document.addEventListener('DOMContentLoaded', function() {
    loadFormData();
    loadCOA();
});
</script>

<?php include '../views/layouts/footer.php'; ?>